// BUG: fully qualified names for methods aren't the only way to refer to a method
//      can't tell the difference between static method call Integer.parseInt
//      and instance method call (e.g., if Integer is the name of a reference variable)
//      overloading, promotion, subtyping, checkstyle is not type aware
//      https://github.com/checkstyle/checkstyle/issues/535

package edu.princeton.cs.lift.checkstyle;

import com.puppycrawl.tools.checkstyle.api.*;

public class IllegalMethodCallCheck extends AbstractCheck {

    /** The regexp for the called method (default is empty). */
    private String format = "$.";

    /** The regexp for the calling method (null means ban all matching method calls regardless of context). */
    private String callingMethodPattern = null;

    /**
     * A key is pointing to the warning message text in "messages.properties"
     * file.
     */
    public static final String MSG = "illegal.method.call";
    public static final String MSG_CALLING_METHOD = "illegal.method.call.from";

    @Override
    public int[] getDefaultTokens() {
        return new int[] { TokenTypes.METHOD_CALL, TokenTypes.LITERAL_NEW };
    }

    @Override
    public int[] getRequiredTokens() {
        return new int[] { TokenTypes.METHOD_CALL, TokenTypes.LITERAL_NEW };
    }

    @Override
    public int[] getAcceptableTokens() {
        return new int[] { TokenTypes.METHOD_CALL, TokenTypes.LITERAL_NEW };
    }


    /**
     * Set the format of the regular expression to match.
     * @param format the format of the regular expression to match.
     */
    public void setFormat(String format) {
        this.format = format;
    }

    /**
     * Set the name of the calling method
     * @param callingMethodPattern the regexp for the calling method
     */
    public final void setCallingMethodPattern(String callingMethodPattern) {
        this.callingMethodPattern = callingMethodPattern;
    }

    // returns whether a node is contained in a calling method
    private boolean isInCallingMethod(DetailAST ast) {
        if (callingMethodPattern == null) return true;
        for (DetailAST x = ast; x != null; x = x.getParent()) {
            if (x.getType() == TokenTypes.METHOD_DEF || x.getType() == TokenTypes.CTOR_DEF) {
                DetailAST ident = x.findFirstToken(TokenTypes.IDENT);
                return ident.getText().matches(callingMethodPattern);
            }
        }
        return false;
    }

    // returns the name of the calling method
    private String getCallingMethodName(DetailAST ast) {
        for (DetailAST x = ast; x != null; x = x.getParent()) {
            if (x.getType() == TokenTypes.METHOD_DEF || x.getType() == TokenTypes.CTOR_DEF) {
                DetailAST ident = x.findFirstToken(TokenTypes.IDENT);
                if (ident.getText().matches(callingMethodPattern))
                    return ident.getText();
            }
        }
        return null;
    }

    @Override
    public void visitToken(DetailAST ast) {
        // don't bother unless we're in the calling method (or no calling method is specified)
        if (!isInCallingMethod(ast)) return;

        // could be Integer.parseInt or p.distanceTo
        if (ast.findFirstToken(TokenTypes.DOT) != null) {
            DetailAST dot = ast.findFirstToken(TokenTypes.DOT);
            FullIdent fullIdent = FullIdent.createFullIdent(dot);
            DetailAST shortIdent = Utilities.findLastToken(dot, TokenTypes.IDENT);
            String fullMethodName = fullIdent.getText();
            String shortMethodName = shortIdent.getText();
            if (fullMethodName.matches(format)) {
                // reportError(ast, fullMethodName);
                reportError(shortIdent, fullMethodName);
            }
            else if (shortMethodName.matches(format)) {
                // reportError(ast, shortMethodName);
                reportError(shortIdent, shortMethodName);
            }
        }

        // calling a static or instance method in the same class, or a constructor
        else {
            DetailAST ident = ast.findFirstToken(TokenTypes.IDENT);

            // creating an array (not calling a constructor)
            if (ident == null) return;

            String methodName = ident.getText();
            if (methodName.matches(format)) {
                reportError(ident, methodName);
                // reportError(ast, methodName);
            }
        }
    }

    private void reportError(DetailAST ast, String calledMethodName) {
        if (callingMethodPattern == null) {
            log(ast,
                MSG,
                calledMethodName + "()");
        }

        else {
            String callingMethodName = getCallingMethodName(ast);
            log(ast,
                MSG_CALLING_METHOD,
                calledMethodName + "()",
                callingMethodName + "()");
        }
    }
}
